import { registerPlugin } from '@wordpress/plugins';
import { PluginDocumentSettingPanel } from '@wordpress/editor';
import { useSelect } from '@wordpress/data';
import { useEntityProp } from '@wordpress/core-data';
import {
	__experimentalVStack as VStack,
	__experimentalNumberControl as NumberControl,
	TextControl,
	ToggleControl,
	SelectControl
} from '@wordpress/components';

const StingSettingPanel = () => {
	const postType = useSelect(
		(select) => select('core/editor').getCurrentPostType(),
		[]
	);

	if (postType !== 'post') {
		return null;
	};

	const [meta, setMeta] = useEntityProp('postType', postType, 'meta');

	const episodeOrRuntime = meta['episode_or_runtime'] || '';
	const releaseYear = meta['release_year'] || '';
	const releaseStatus = meta['release_status'] || '';
	const movieQuality = meta['movie_quality'] || '';

	const updateEpisodeOrRuntime = (newValue) => {
		setMeta({ ...meta, episode_or_runtime: newValue });
	};

	const updateReleaseYear = (newValue) => {
		setMeta({ ...meta, release_year: newValue });
	};

	const updateReleaseStatus = (isCompleted) => {
		setMeta({ ...meta, release_status: isCompleted ? 'completed' : '' });
	};

	const updateMovieQuality = (newValue) => {
		setMeta({ ...meta, movie_quality: newValue });
	};

	const isCompleted = releaseStatus === 'completed';
	const toggleLabel = isCompleted ? 'Completed' : 'Mark as Completed';
	const toggleHelpText = isCompleted ? "This drama has finished airing." : "Toggle to mark the drama as finished.";

	const isNumeric = /^\d+$/.test(episodeOrRuntime);
	const showStatusToggle = isNumeric && episodeOrRuntime && episodeOrRuntime !== '0';
	const showMovieQuality = episodeOrRuntime && !isNumeric;

	return (
		<PluginDocumentSettingPanel
			name="custom-label-panel"
			title="Custom Label Settings"
			className="custom-label-panel"
		>
			<VStack>
				<TextControl
					label="Episode Number or Video Runtime"
					type="text"
					value={ episodeOrRuntime }
					onChange={ updateEpisodeOrRuntime }
					help="For drama, enter the episode number (e.g., 26). For movie, enter the runtime (e.g., 2h 45m)."
				/>
				<NumberControl
					label="Release Year"
					value={ releaseYear }
					onChange={ updateReleaseYear }
					placeholder="2026"
				/>
				{ showStatusToggle && (
					<ToggleControl
						label={ toggleLabel }
						checked={ isCompleted }
						onChange={ updateReleaseStatus }
						help={ toggleHelpText }
					/>
				) }
				{ showMovieQuality && (
					<SelectControl
						label="Quality"
						value={ movieQuality }
						options={ [
							{ label: '— Select Quality —', value: '' },
							{ label: 'CAM - Low Quality', value: 'cam' },
							{ label: 'HD - 720p', value: 'hd' },
							{ label: 'FHD - 1080p', value: 'fhd' }
						] }
						onChange={ updateMovieQuality }
					/>
				) }
			</VStack>
		</PluginDocumentSettingPanel>
	)
};

registerPlugin( 'sting-document-setting-panel', {
	render: StingSettingPanel,
} );