import { registerPlugin } from '@wordpress/plugins';
import { PluginDocumentSettingPanel } from '@wordpress/editor';
import { useSelect } from '@wordpress/data';
import { useEntityProp } from '@wordpress/core-data';
import {
	__experimentalVStack as VStack,
	__experimentalNumberControl as NumberControl,
	ToggleControl,
	SelectControl
} from '@wordpress/components';

const SpringSettingPanel = () => {
	const postType = useSelect(
		(select) => select('core/editor').getCurrentPostType(),
		[]
	);

	if (postType !== 'post') {
		return null;
	};

	const [meta, setMeta] = useEntityProp('postType', postType, 'meta');

	const episodeNumber = meta['episode_number'] || '';
	const releaseStatus = meta['release_status'] || '';
	const movieQuality = meta['movie_quality'] || '';

	const updateEpisodeNumber = (newValue) => {
		setMeta({ ...meta, episode_number: newValue });
	};

	const updateReleaseStatus = (isCompleted) => {
		setMeta({ ...meta, release_status: isCompleted ? 'completed' : '' });
	};

	const updateMovieQuality = (newValue) => {
		setMeta({ ...meta, movie_quality: newValue });
	};

	const isCompleted = releaseStatus === 'completed';
	const toggleLabel = isCompleted ? 'Completed' : 'Mark as Completed';
	const toggleHelpText = isCompleted ? "This drama has finished airing." : "Toggle to mark the drama as finished.";

	const isMovie = movieQuality !== '';
	const showMovieQuality = !episodeNumber;

	return (
		<PluginDocumentSettingPanel
			name="custom-label-panel"
			title="Custom Label Settings"
			className="custom-label-panel"
		>
			<VStack>
				{ !isMovie && (
					<>
						<NumberControl
							label="Episode Number"
							value={ episodeNumber }
							onChange={ updateEpisodeNumber }
							placeholder="0"
						/>
						<ToggleControl
							label={ toggleLabel }
							checked={ isCompleted }
							onChange={ updateReleaseStatus }
							help={ toggleHelpText }
						/>
					</>
				) }
				{ showMovieQuality && (
					<SelectControl
						label="Quality"
						value={ movieQuality }
						options={ [
							{ label: '— Select Quality —', value: '' },
							{ label: 'CAM - Low Quality', value: 'cam' },
							{ label: 'HD - 720p', value: 'hd' },
							{ label: 'FHD - 1080p', value: 'fhd' }
						] }
						onChange={ updateMovieQuality }
					/>
				) }
			</VStack>
		</PluginDocumentSettingPanel>
	)
};

registerPlugin( 'spring-document-setting-panel', {
	render: SpringSettingPanel,
} );